<?php

/**
 * The channel object model.
 *
 * @since		1.0.0
 *
 * @package		TVindoor
 * @subpackage	TVindoor/includes
 * @author		Menno Luitjes | Adapted by Valmy Gomes
 */
class TVindoor_Channel {
    	/**
	 * The TVindoor Radio post type name.
	 *
	 * @since    2.0.0
	 * @access   private
	 * @var      string    $post_type_name    The TVindoor Radio post type name.
	 * Valmy Gomes em 23/07/2021
	 */
    
    const post_type_name_radio = 'tvindoor_radio';
    const post_type_name_playlist = 'tvindoor_playlist';

	/**
	 * The TVindoor Channel post type name.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $post_type_name    The TVindoor Channel post type name.
	 */
	const post_type_name = 'tvindoor_channel';

	public $ID;
	private $post;

	/**
	 * The slides of this channel.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $slides    The slides of this channel.
	 */
	private $slides;

	/**
	 * The slides duration setting of this channel.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $slides    The slides duration setting of this channel.
	 */
	private $slides_duration;

	/**
	 * The slides transition setting of this channel.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $slides    The slides transition setting of this channel.
	 */
	private $slides_transition;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since	1.0.0
	 * @param	int or WP_Post	$ID		The id or the WP_Post object of the channel.
	 */
	public function __construct( $ID = false ) {

		if ( $ID instanceof WP_Post ) {
			// $ID is a WP_Post object
			$this->post = $ID;
			$ID = $ID->ID;
		}

		$this->ID = $ID;
	}

	/**
	 * Outputs the channel classes for use in the template.
	 *
	 * The output is escaped, so this method can be used in templates without further escaping.
	 *
	 * @since	1.0.1			Escaped the output.
	 *
	 * @param 	array 	$classes
	 * @return 	string
	 */
	public function classes( $classes = array() ) {

		$classes[] = 'tvindoor-channel';
		$classes[] = 'tvindoor-channel-' . intval( $this->ID );
		$classes[] = 'tvindoor-transition-' . $this->get_slides_transition();

		if ( empty( $classes ) ) {
			return;
		}

		?> class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>" <?php
	}

	/**
	 * Get slides for this channel.
	 *
	 * @since	1.0.0
	 * @since	1.3.2	Only includes slides that are published.
	 *
	 * @access	public
	 * @return	array of TVindoor_Slide	The slides for this channel.
	 */
	public function get_slides() {

		if ( ! isset( $this->slides ) ) {

			$slides = array();

			$posts = get_post_meta( $this->ID, TVindoor_Slide::post_type_name, true );

			if ( ! empty( $posts ) ) {
				foreach ( $posts as $post ) {

					// Only include slides with post status 'publish'
					if ( 'publish' != get_post_status( $post ) ) {
						continue;
					}

					$slide = new TVindoor_Slide( $post );
					$slides[] = $slide;
				}
			}

			$this->slides = $slides;
		}

		return $this->slides;
	}

	/**
	 * Get slides duration setting for this channel as saved in the database.
	 *
	 * @since	1.0.0
	 * @access	public
	 * @return	string		The slides duration setting for this channel as saved in the database.
	 */
	public function get_saved_slides_duration() {
		return get_post_meta( $this->ID, TVindoor_Channel::post_type_name . '_slides_duration', true );
	}

	/**
	 * Get slides transition setting for this channel as saved in the database.
	 *
	 * @since	1.0.0
	 * @access	public
	 * @return	string		The slides transition setting for this channel as saved in the database.
	 */
	public function get_saved_slides_transition() {
			return get_post_meta( $this->ID, TVindoor_Channel::post_type_name . '_slides_transition', true );
	}

	/**
	 * Get slides duration setting for this channel, or the default slides duration when not set.
	 *
	 * @since	1.0.0
	 * @access	public
	 * @return	string		The slides duration setting for this channel, or the default slides duration when not set.
	 */
	public function get_slides_duration() {

		if ( ! isset( $this->slides_duration ) ) {

			$slides_duration = self::get_saved_slides_duration();
			if ( empty( $slides_duration ) ) {
				$slides_duration = TVindoor_Slides::get_default_slides_duration();
			}
			$this->slides_duration = $slides_duration;
		}

		return $this->slides_duration;
	}

	/**
	 * Get slides transition setting for this channel, or the default slides transition when not set.
	 *
	 * @since	1.0.0
	 * @access	public
	 * @return	string		The slides transition setting for this channel, or the default slides transition when not set.
	 */
	public function get_slides_transition() {

		if ( ! isset( $this->slides_transition ) ) {

			$slides_transition = self::get_saved_slides_transition();
			if ( empty( $slides_transition ) ) {
				$slides_transition = TVindoor_Slides::get_default_slides_transition();
			}
			$this->slides_transition = $slides_transition;
		}

		return $this->slides_transition;
	}
}
